#!/bin/bash

###############################################################################################################################
# Installation script for KeyTalk Linux client
###############################################################################################################################

set -o errexit
set -o nounset

# Duplicate all the output to the log file, handy for troubleshooting
# redirect stdout ( > ) into a named pipe ( >() ) running "tee"
exec > >(tee -a -i /var/log/ktinstaller.log)
# include strerr
exec 2>&1

IS_APACHE_CERT_RENEWAL_PREREQUISITES_OK=false
IS_TOMCAT_CERT_RENEWAL_PREREQUISITES_OK=false

function usage()
{
    echo "Usage: $0 remove            uninstall KeyTalk"
    echo "Usage: $0                   install KeyTalk client without customizing."
    exit 2
}

function log_debug()
{
    echo "[$(date)] $1"
}

function log_error()
{
    echo "[$(date)] [ERROR] $1" 1>&2
}

function log_warning()
{
    echo "[$(date)] [WARNING] $1" 1>&2
}

function distro_spec()
{
    local distro_name=$(lsb_release --id --short | tr "[:upper:]" "[:lower:]")
    local distro_version_major=$(lsb_release --release --short | egrep -o [0-9]+ | sed -n '1p')
    echo ${distro_name}${distro_version_major}
}

function platform_info()
{
    echo $(lsb_release --description --short) $(dpkg --print-architecture)
}

function initialize()
{
    if [ "$(id -u)" != "0" ]; then
        log_error "KeyTalk agent installation script must be run as root"
        exit 1
    fi
    if apt show -qq keytalk >/dev/null 2>&1 ; then
        log_error "KeyTalk agent system package is installed. Please use 'apt' to manage KeyTalk installation."
        exit 1
    fi
    log_debug "Starting KeyTalk agent installation..."
}

function need_unintstall_existing_keytalk()
{
    if [ -f /etc/keytalk/version ]; then
        if grep  -q 'version\=4\.' /etc/keytalk/version ; then
            return 0
        fi
    fi
    return 1
}

function install_python_openssl_binding()
{
    local distro_name=$(lsb_release --id --short | tr "[:upper:]" "[:lower:]")
    local distro_version_major=$(lsb_release --release --short | egrep -o [0-9]+ | sed -n '1p')

    if [[ "${distro_name}" == "ubuntu" ]]; then
        if (( distro_version_major <= 20 )); then
            #@notice pyOpenSSL prior to v20 has a bug causing errors like <class 'AttributeError'> module 'lib' has no attribute 'X509_get_notBefore'
            # pip repo is known to contain more recent packages compared to the Debian upstream repos (e.g. pyOpenSSL from Ubuntu 20 repos contains dated pyOpenSSL v19)
            #@notice make sure to install the latest version of pip because out-of-date pip is the major source of package installation errors
            #@notice that pip3 might reside under /usr/local/bin which might not be in $PATH; so should either call pip3 with full path or use python3 -m pip
            DEBIAN_FRONTEND=noninteractive apt-get -q -y install python3-pip
            python3 -m pip install --upgrade pip
            python3 -m pip install -U pyOpenSSL
        elif (( distro_version_major >= 22 )); then
            apt-get -q -y install python3-openssl
        fi
    fi
}

function install_python()
{
    if type python > /dev/null 2>&1 ; then
        DEBIAN_FRONTEND=noninteractive apt-get -qq -y install python3-lxml
        install_python_openssl_binding
    else
        log_debug "Installing Python 3"
        DEBIAN_FRONTEND=noninteractive apt-get -qq -y install python3 python3-lxml
        install_python_openssl_binding
        [ ! -f /usr/bin/python ] && update-alternatives --install /usr/bin/python python /usr/bin/python3 1
    fi
}

function uninstall_keytalk_with_self_installer()
{
    bash .uninstaller
}

function uninstall_keytalk()
{
    if [ -x "/usr/local/bin/keytalk/uninstall_keytalk" ] ; then
        /usr/local/bin/keytalk/uninstall_keytalk
    else
       uninstall_keytalk_with_self_installer
    fi
}

function install_keytalk()
{
    log_debug "Installing KeyTalk on $(platform_info)..."

    local keytalk_uuid_filepath='/etc/.keytalk_uuid'
    if [ ! -f "${keytalk_uuid_filepath}" ]; then
        log_debug "    Generating and installing random uuid"
        log_debug $(cat /proc/sys/kernel/random/uuid | tr -d '-') >> "${keytalk_uuid_filepath}"
    fi

    mkdir -p /usr/local/bin/keytalk
    cp ktclient ktconfig ktconfupdater ktconfigtool ktprgen hwutils /usr/local/bin/keytalk/
    cp .uninstaller /usr/local/bin/keytalk/uninstall_keytalk
    chmod +x /usr/local/bin/keytalk/uninstall_keytalk
    # set setuid bit on ktconfig to let it alter /etc/keytalk
    chown root /usr/local/bin/keytalk/ktconfig
    chmod u+s /usr/local/bin/keytalk/ktconfig
    # set setuid bit on hwutils to let it read system serial
    chown root /usr/local/bin/keytalk/hwutils
    chmod u+s /usr/local/bin/keytalk/hwutils

    mkdir -p /usr/local/lib/keytalk
    cp *.so /usr/local/lib/keytalk/
    # regardless -Wl,-R,/usr/local/lib/keytalk linker option (-Wl,-rpath) applied to KeyTalk executables, the call to ldconfig is still needed, at least to locate TSS libs
    ldconfig /usr/local/lib/keytalk/

    mkdir -p /etc/keytalk
    cp resept.ini version cert-renewal.ini /etc/keytalk/
    if [ -d .keytalk/ ] ; then
        # prepare the initial user configuration to be later populated by KT apps (typically ktconfig) to initialize user
        cp -rf .keytalk/ /etc/keytalk/
    fi
    /usr/local/bin/keytalk/ktconfupdater --set-install-dir /usr/local/bin/keytalk

    mkdir -p /usr/share/doc/keytalk/
    cp KeyTalk_LinuxClient.txt /usr/share/doc/keytalk/
    cp KeyTalk_LinuxClient.pdf /usr/share/doc/keytalk/

    if ${IS_APACHE_CERT_RENEWAL_PREREQUISITES_OK} ; then
        log_debug "    Installing KeyTalk Apache certificate renewal..."
        cp renew_apache_ssl_cert util.py apache_util.py /usr/local/bin/keytalk/
        # notice that cron ignores cron jobs having an extension in a filename
        cp etc_cron.d_keytalk-apache /etc/cron.d/keytalk-apache
        chmod 644 /etc/cron.d/keytalk-apache

        mkdir -p /etc/keytalk
        cp apache.ini /etc/keytalk/

        cp KeyTalk_LinuxClient_for_Apache.txt /usr/share/doc/keytalk/
        cp KeyTalk_LinuxClient_for_Apache.pdf /usr/share/doc/keytalk/

        if [ -f /etc/httpd/conf.d/ssl.conf ] ; then
            # Remove spaces between the last 2 lines of "CustomLog" making it a single-line command.
            # Without this fix our parser can't handle this file correctly
            sed -i -e '{N; s/.*CustomLog.*\n.*"%t.*/CustomLog logs\/ssl_request_log "%t %h %{SSL_PROTOCOL}x %{SSL_CIPHER}x \\"%r\\" %b"/g}' /etc/httpd/conf.d/ssl.conf
        fi
    fi

    if ${IS_TOMCAT_CERT_RENEWAL_PREREQUISITES_OK} ; then
        log_debug "    Installing KeyTalk Tomcat certificate renewal..."
        cp renew_tomcat_ssl_cert util.py tomcat_util.py tomcat.sh /usr/local/bin/keytalk/
        # notice that cron ignores cron jobs having an extension in a filename
        cp etc_cron.d_keytalk-tomcat /etc/cron.d/keytalk-tomcat
        chmod 644 /etc/cron.d/keytalk-tomcat

        mkdir -p /etc/keytalk
        cp tomcat.ini /etc/keytalk/

        cp KeyTalk_LinuxClient_for_Tomcat.txt /usr/share/doc/keytalk/
        cp KeyTalk_LinuxClient_for_Tomcat.pdf /usr/share/doc/keytalk/
    fi

    install_keytalk_ca_updater_service
    configure_tpm
    log_debug "Installation complete. Please customize KeyTalk by calling /usr/local/bin/keytalk/ktconfig --rccd-path <rccd-url>"
}

# Check if an executable can be found in the path
function has_executable()
{
    if type "$1" > /dev/null 2>&1 && [ -x "$(type -p $1)" ]; then
        return 0
    else
        return 1
    fi
}

function is_tomcat_installed()
{
    for probe in tomcat10 tomcat9 tomcat8 tomcat
    do
        if [[ -f "/usr/lib/systemd/system/${probe}.service" || -f "/etc/systemd/system/${probe}.service" || -f "/etc/init.d/${probe}" ]] ; then
            return 0
        fi
    done
    return 1
}

function check_platform_compatibility()
{
    local os=$(uname)
    if [ x"${os}" != x"Linux" ]; then
        log_error "KeyTalk Linux client requires Linux to install, but '${os}' detected"
        return 1
    fi

    local current_arch=$(dpkg --print-architecture)
    if [ x"${current_arch}" != x"amd64" ]; then
        log_error "KeyTalk Linux client requires 64-bit Linux to install"
        return 1
    fi

    if ! has_executable lsb_release ; then
        apt-get -qq -y update
        apt-get -y install lsb-release
    fi

    local current_distro_name=$(lsb_release --id --short | tr "[:upper:]" "[:lower:]")
    local current_distro_version_major=$(lsb_release --release --short | egrep -o [0-9]+ | sed -n '1p')
    # 'compatible-platform' file contains <distro-name>-<distro-version-major>-<distro-current_arch> e.g. "ubuntu-22-amd64"
    local required_distro_name=$(cat ./compatible-platform | cut -d '-' -f 1)
    local required_distro_version_major=$(cat ./compatible-platform | cut -d '-' -f 2)
    local required_arch=$(cat ./compatible-platform | cut -d '-' -f 3)

    if ! has_executable inotifywait ; then
         apt-get -q -y install inotify-tools
    fi


    #
    # Check current platform matches the platform required by the client installation package
    #
    if [ x"${current_distro_name}" != x"${required_distro_name}" ]; then
        log_error "KeyTalk Linux client requires Linux ${required_distro_name} to install but ${current_distro_name} found"
        return 1
    fi
    if [ x"${current_distro_version_major}" != x"${required_distro_version_major}" ]; then
        log_error "KeyTalk Linux client requires Linux ${required_distro_name} ${required_distro_version_major} to install but version ${current_distro_version_major} found"
        return 1
    fi
    if [ x"${current_arch}" != x"${required_arch}" ]; then
        log_error "KeyTalk Linux client requires Linux with ${required_arch} architecture to install"
        return 1
    fi

    #
    # Check the platform the platform required by the client installation package is supported
    #
    local required_distro_name_version="${required_distro_name}${required_distro_version_major}"
    if [[ "${required_distro_name_version}" =~ ^(ubuntu20|ubuntu22|ubuntu24)$ ]] ; then
        return 0 # ok
    else
        log_error "KeyTalk client is built for Linux ${required_distro_name_version} which is not supported"
        return 1
    fi
}

function install_apache_cert_renewal_prerequisities()
{
    # Apache version should be 2.2 - 2.4
    if ! has_executable apache2 ; then
        log_debug "KeyTalk Apache SSL certificate renewal feature will not be installed because no Apache 2 installation detected."
        return 0
    fi

    local apache_version=( $(apache2 -v | grep "Server version" | egrep -o [0-9]+) )

    if [ ${apache_version[0]} -ne 2 ] ; then
        log_warning "KeyTalk Apache SSL certificate renewal feature will not be installed because Apache ${apache_version} is not supported by KeyTalk client certificate renewal. Apache 2.2-2.4 is required."
        return 0
    fi
    if [ ${apache_version[1]} -ne 2 -a ${apache_version[1]} -ne 4 ] ; then
        log_warning "KeyTalk Apache SSL certificate renewal feature will not be installed because Apache ${apache_version} is not supported by KeyTalk client certificate renewal. Apache 2.2-2.4 is required."
        return 0
    fi

    log_debug "Installing Apache certificate renewal prerequisites for KeyTalk on $(platform_info)"
    DEBIAN_FRONTEND=noninteractive apt-get -qq -y install cron
    install_python
    IS_APACHE_CERT_RENEWAL_PREREQUISITES_OK=true
    return 0
}

function install_tomcat_cert_renewal_prerequisities()
{
    if ! is_tomcat_installed ; then
        log_debug "KeyTalk Tomcat SSL certificate renewal will not be installed because no Tomcat installation detected."
        return 0
    fi

    log_debug "Installing Tomcat certificate renewal prerequisites for KeyTalk on $(platform_info)"
    install_python
    IS_TOMCAT_CERT_RENEWAL_PREREQUISITES_OK=true
    return 0
}

function install_tpm_prerequisities()
{
    log_debug "Installing TPM prerequisites for KeyTalk on $(platform_info)"
    apt-get -qq -y install libssl-dev libjson-c-dev libini-config-dev libltdl-dev libsqlite3-dev
}

function configure_tpm()
{
    log_debug "Adding TPM user group for KeyTalk on $(platform_info)"
    local cur_user=${SUDO_USER:-$(whoami)}
    groupadd -f tss
    usermod -aG tss ${cur_user}
}


# Check and install prerequisites
function install_prerequisites()
{
    check_platform_compatibility

    log_debug "Installing prerequisites for KeyTalk on $(platform_info)"
    apt-get -qq -y update
    apt-get -y install ca-certificates hdparm psmisc libyaml-0-2 libyaml-cpp0.* libnss3-tools libkrb5-3 libcurl4-openssl-dev libconfig++-dev libboost-all-dev
    install_apache_cert_renewal_prerequisities
    install_tomcat_cert_renewal_prerequisities
    install_tpm_prerequisities
}

function install_keytalk_ca_updater_service()
{
    local distro_version_major=$(lsb_release --release --short | egrep -o [0-9]+ | sed -n '1p')

    log_debug "    Installing KeyTalk CA Updater service"
    cp keytalk_ca_updater.sh /usr/local/bin/keytalk/

    if type systemctl > /dev/null 2>&1 ; then
        cp -f etc_systemd_system_keytalk-ca-updater.service /etc/systemd/system/keytalk-ca-updater.service
        systemctl daemon-reload
        systemctl restart keytalk-ca-updater.service
        systemctl enable keytalk-ca-updater.service
    else
        # Use legacy systemV-compatible service
        cp -f etc_init.d_keytalk-ca-updater /etc/init.d/keytalk-ca-updater
        service keytalk-ca-updater restart
        chkconfig keytalk-ca-updater on
    fi

}

function on_keytalk_installation_error()
{
    log_warning "KeyTalk installation failed, rolling back"
    uninstall_keytalk_with_self_installer
    exit 1
}

#
# Entry point
#

cd `dirname "$0"` # Go to current script directory

initialize

if [ $# -eq 0 ]; then
    if need_unintstall_existing_keytalk; then
        log_warning "Existing KeyTalk agent installation is not compatible with the current. Uninstalling it first ..."
        uninstall_keytalk
    fi
    install_prerequisites
    trap on_keytalk_installation_error EXIT
    install_keytalk
    trap "" EXIT
elif [ $# -eq 1 -a x"$1" == x"remove" ]; then
    uninstall_keytalk
else
    usage
fi
